/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   TCPIP Resource Framework API Header File
::   Copyright   :   (C)2004-2009 Woodward
::   Platform(s) :   MPC5xx
::   Dependants  :   MPC5xx OS sub-project
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource_TCPIP.h
    \brief This is a port of lwIP which is an open source implementation of the TCP/IP protocol 
    suite that was originally written by Adam Dunkels of the Swedish Institute of Computer Science
*/

#ifndef __RESOURCE_TCPIP_H
#define __RESOURCE_TCPIP_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <MacroDefn.h>
#include <resource.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/
/*! Define a IPV4 address in platform specific endian */
#define IPV4_ADDRESS(a,b,c,d) ( ((((uint4)(a)) & 0xFFL) << 24) + ((((uint4)(b)) & 0xFFL) << 16) +  \
                                ((((uint4)(c)) & 0xFFL) <<  8) + ((((uint4)(d)) & 0xFFL)      )    )


/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/

#pragma pack(1)

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid TCPIP
           attributes in the S_TCPIPCreateAttributes data structure.

Each mask definition applies to a particular member of one of the data structures.
\code
    ...
    S_TCPIPCreateAttributes AttribsObj;

    S_TCPIPNetworkInterface InterfaceArr[1];
    InterfaceArr.eInterfaceType = NETIF_SLIP;
    InterfaceArr.NetIFSlip.ePort = RES_RS232;
    InterfaceArr.NetIFSlip.u4BaudRate = 57600;

    AttribsObj.uValidAttributesMask = USE_TCPIP_HTTPD;
    AttribsObj.pNetworkInterfaceArr = InterfaceArr;
    AttribsObj.uNumNetworkInterfaces = CountOf(InterfaceArr);
    ... \endcode */
typedef enum
{
/*! Include this mask in S_TCPIPCreateAttributes::uValidAttributesMask to enable the HTTP Server on the stack */
    USE_TCPIP_HTTPD = 0x01,
    USE_TCPIP_FTPD = 0x02,
} E_TCPIPCreateAttributeMask;

typedef uint1 TCPIPCreateAttributesMask_U;

/*! The available interfaces this stack supports */
typedef enum
{
    NETIF_SLIP,             /*!< Tunnels TCP/IP over SerialLineIP on a serial port */
    NETIF_SMSC91C111,       /*!< Uses a SMSC 91C111 ethernet controller */
    NUM_SUPPORTED_INTERFACES
} E_InterfaceType;

/*! The settings for a SLIP network interface */
typedef struct
{
    E_ModuleResource ePort; /*!< The \ref BEHAVIOUR_COMM compatible serial resource to use for SLIP */
    uint4 u4BaudRate;       /*!< The baud rate to run at */
} S_NetIFSlip;

/*! The settings for a SMC91C111 ethernet controller */
typedef struct
{
    uint4 ToDo;
} S_NetIF91C111;

/*! The TCPIP network interface setup */
typedef struct
{
    /*!< Selects one of the union items for individual settings */
    E_InterfaceType eInterfaceType;
    /*! Fill in the relevant settings according to the interface type */
    union
    {
        /*!< Fill in when S_TCPIPNetworkInterface::eInterfaceType is \ref NETIF_SLIP */
        S_NetIFSlip NetIFSlip;
        /*!< Fill in when S_TCPIPNetworkInterface::eInterfaceType is \ref NETIF_SMSC91C111 */
        S_NetIF91C111 NetIF91C111;
    };
    /*! The platform specific network interface address (i.e. natural endian of the target) */
    uint4 u4NetAddress;
    /*! The platform specific subnet mask (i.e. natural endian of the target) */
    uint4 u4NetMask;
    /*! The platform specific gateway address (i.e. natural endian of the target) */
    uint4 u4Gateway;
} S_TCPIPNetworkInterface;

/*! Describes the creation attributes for a resource with the \ref BEHAVIOUR_TCPIP */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_TCPIP_HTTPD etc] that are valid for this instance of the data structure. 
    The framework will employ the creation defaults for those attributes that are left undefined. */
    TCPIPCreateAttributesMask_U uValidAttributesMask;
/*! An array of length S_TCPIPCreateAttributes::uNumNetworkInterfaces interfaces to open and register with the stack.  
    See S_TCPIPNetworkInterface */
    S_TCPIPNetworkInterface* pNetworkInterfaceArr;
/*! The number of interfaces contained in S_TCPIPCreateAttributes::pNetworkInterfaceArr */
    NativeVar_U uNumNetworkInterfaces;

} S_TCPIPCreateAttributes;

typedef S_TCPIPCreateAttributes const* S_TCPIPCreateAttributesPtr;

#pragma pack()

/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/
NativeError_S CreateResourceBEHAVIOUR_TCPIP(E_ModuleResource in_eResource, S_TCPIPCreateAttributes const* in_pAttribObj);

#endif /* __RESOURCE_TCPIP_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
